<?php

namespace App\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use App\Models\SpamAttempt;

class SpamDetectionService
{
    /**
     * Check if the submission is spam
     */
    public function isSpam(Request $request, string $formType = 'contact'): bool
    {
        // Check honeypot field
        if ($this->checkHoneypot($request)) {
            $this->logSpamAttempt($request, $formType, 'honeypot', 'Honeypot field filled');
            return true;
        }

        // Check submission time (should take at least 3 seconds)
        if ($this->checkSubmissionTime($request)) {
            $this->logSpamAttempt($request, $formType, 'timing', 'Form submitted too quickly');
            return true;
        }

        // Check for spam keywords
        if ($this->checkSpamKeywords($request)) {
            $this->logSpamAttempt($request, $formType, 'keywords', 'Spam keywords found');
            return true;
        }

        // Check for suspicious patterns
        if ($this->checkSuspiciousPatterns($request)) {
            $this->logSpamAttempt($request, $formType, 'patterns', 'Suspicious patterns found');
            return true;
        }

        // Check IP reputation
        if ($this->checkIpReputation($request)) {
            $this->logSpamAttempt($request, $formType, 'ip_reputation', 'IP reputation check failed');
            return true;
        }

        return false;
    }

    /**
     * Check if honeypot field is filled (bots fill it, humans don't)
     */
    private function checkHoneypot(Request $request): bool
    {
        $honeypotFields = ['website', 'url', 'phone', 'company', 'subject'];
        
        foreach ($honeypotFields as $field) {
            if ($request->filled($field) && !empty($request->input($field))) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if form was submitted too quickly (less than 3 seconds)
     */
    private function checkSubmissionTime(Request $request): bool
    {
        $formStartTime = $request->input('form_start_time');
        
        if (!$formStartTime) {
            return true; // No start time, likely spam
        }

        $submissionTime = time();
        $timeDiff = $submissionTime - $formStartTime;

        return $timeDiff < 3; // Less than 3 seconds is suspicious
    }

    /**
     * Check for common spam keywords
     */
    private function checkSpamKeywords(Request $request): bool
    {
        $spamKeywords = [
            'viagra', 'cialis', 'casino', 'poker', 'loan', 'credit', 'debt',
            'weight loss', 'diet', 'make money', 'earn money', 'work from home',
            'free trial', 'limited time', 'act now', 'click here', 'buy now',
            'cheap', 'discount', 'sale', 'offer', 'promotion', 'bonus',
            'lottery', 'winner', 'prize', 'claim', 'urgent', 'important',
            'sex', 'adult', 'dating', 'meet singles', 'enlarge', 'penis',
            'pharmacy', 'medication', 'prescription', 'drugs', 'pills'
        ];

        $content = strtolower($request->input('message', '') . ' ' . $request->input('full_name', ''));
        
        foreach ($spamKeywords as $keyword) {
            if (str_contains($content, $keyword)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check for suspicious patterns
     */
    private function checkSuspiciousPatterns(Request $request): bool
    {
        $message = $request->input('message', '');
        $email = $request->input('email', '');
        $name = $request->input('full_name', '');

        // Check for excessive capitalization
        if (preg_match('/[A-Z]{5,}/', $message)) {
            return true;
        }

        // Check for excessive punctuation
        if (substr_count($message, '!') > 3 || substr_count($message, '?') > 3) {
            return true;
        }

        // Check for suspicious email patterns
        if (preg_match('/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/', $email)) {
            // Check for disposable email domains
            $disposableDomains = [
                'tempmail.org', 'guerrillamail.com', '10minutemail.com',
                'mailinator.com', 'yopmail.com', 'throwaway.email',
                'temp-mail.org', 'fakeinbox.com', 'getairmail.com',
                'meme20271@outlook.fr', 'ouss7788@gmail.com', // Known spam emails
                'tempmail.net', 'disposable.email', 'throwaway.ml',
                'tempail.com', 'sharklasers.com', 'guerrillamail.net',
                'guerrillamail.org', 'guerrillamail.biz', 'guerrillamail.de',
                'guerrillamail.info', 'guerrillamail.net', 'guerrillamail.org',
                'pokemail.net', 'spam4.me', 'bccto.me', 'chacuo.net',
                'dispostable.com', 'mailnesia.com', 'maildrop.cc',
                'mailcatch.com', 'inboxalias.com', 'mailmetrash.com',
                'trashmail.net', 'trashmail.com', 'trashmail.org',
                'spamgourmet.com', 'spamgourmet.net', 'spamgourmet.org',
                'spam.la', 'binkmail.com', 'bobmail.info', 'chammy.info',
                'devnullmail.com', 'letthemeatspam.com', 'mailin8r.com',
                'mailinator2.com', 'notmailinator.com', 'reallymymail.com',
                'reconmail.com', 'safetymail.info', 'sogetthis.com',
                'spamhereplease.com', 'superrito.com', 'thisisnotmyrealemail.com',
                'tradermail.info', 'veryrealemail.com', 'wegwerfmail.de',
                'wegwerfmail.net', 'wegwerfmail.org', 'wegwerpmailadres.nl',
                'wetrainbayarea.com', 'wetrainbayarea.org', 'wh4f.org',
                'whyspam.me', 'willselfdestruct.com', 'wuzup.net',
                'wuzupmail.net', 'yeah.net', 'yopmail.net', 'yopmail.org',
                'yopmail.pp.ua', 'ypmail.webarnak.fr.eu.org', 'cool.fr.nf',
                'jetable.fr.nf', 'nospam.ze.tc', 'nomail.xl.cx', 'mega.zik.dj',
                'speed.1s.fr', 'courriel.fr.nf', 'moncourrier.fr.nf',
                'monemail.fr.nf', 'monmail.fr.nf', 'test.com', 'example.com',
                'example.org', 'example.net', 'test.org', 'test.net'
            ];
            
            $emailDomain = strtolower(substr(strrchr($email, "@"), 1));
            if (in_array($emailDomain, $disposableDomains)) {
                return true;
            }
        }

        // Check for suspicious name patterns
        if (strlen($name) > 50 || preg_match('/[0-9]{3,}/', $name)) {
            return true;
        }

        return false;
    }

    /**
     * Check IP reputation
     */
    private function checkIpReputation(Request $request): bool
    {
        $ip = $request->ip();
        $cacheKey = 'spam_ip_' . md5($ip);
        
        // Check if IP is already flagged
        if (Cache::has($cacheKey)) {
            return true;
        }

        // Check for localhost or private IPs (suspicious for production)
        if (in_array($ip, ['127.0.0.1', '::1']) || 
            preg_match('/^192\.168\./', $ip) || 
            preg_match('/^10\./', $ip) || 
            preg_match('/^172\.(1[6-9]|2[0-9]|3[0-1])\./', $ip)) {
            return true;
        }

        return false;
    }

    /**
     * Flag an IP as spam
     */
    public function flagSpamIp(string $ip): void
    {
        $cacheKey = 'spam_ip_' . md5($ip);
        Cache::put($cacheKey, true, now()->addDays(7)); // Block for 7 days
    }

    /**
     * Log spam attempt to database
     */
    private function logSpamAttempt(Request $request, string $formType, string $spamType, string $details): void
    {
        try {
            SpamAttempt::create([
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'form_type' => $formType,
                'spam_type' => $spamType,
                'details' => $details,
                'blocked' => true,
                'blocked_at' => now(),
            ]);
            
            // Immediately flag the IP as spam for 7 days
            $this->flagSpamIp($request->ip());
            
        } catch (\Exception $e) {
            Log::error('Failed to log spam attempt: ' . $e->getMessage());
        }
    }

    /**
     * Get form start time for timing validation
     */
    public function getFormStartTime(): int
    {
        return time();
    }
}
