<?php

namespace App\Services;

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\File;

class SitemapService
{
    protected $urls = [];
    protected $baseUrl;

    public function __construct()
    {
        $this->baseUrl = config('app.url');
    }

    public function addUrl($url, $priority = 0.5, $changefreq = 'weekly', $lastmod = null)
    {
        $this->urls[] = [
            'url' => $url,
            'priority' => $priority,
            'changefreq' => $changefreq,
            'lastmod' => $lastmod ?: now()->toISOString()
        ];

        return $this;
    }

    public function addStaticPages()
    {
        $staticPages = [
            '/' => ['priority' => 1.0, 'changefreq' => 'daily'],
            '/about-us' => ['priority' => 0.8, 'changefreq' => 'monthly'],
            '/our-brands' => ['priority' => 0.9, 'changefreq' => 'weekly'],
            '/news-room' => ['priority' => 0.8, 'changefreq' => 'daily'],
            '/careers' => ['priority' => 0.7, 'changefreq' => 'weekly'],
            '/contact' => ['priority' => 0.6, 'changefreq' => 'monthly'],
            '/investor-relations' => ['priority' => 0.8, 'changefreq' => 'weekly'],
            '/sustainability' => ['priority' => 0.8, 'changefreq' => 'monthly'],
            '/our-leadership' => ['priority' => 0.7, 'changefreq' => 'monthly'],
            '/our-heritage' => ['priority' => 0.6, 'changefreq' => 'yearly'],
            '/operations-structure' => ['priority' => 0.7, 'changefreq' => 'monthly'],
            '/virtual-tour' => ['priority' => 0.6, 'changefreq' => 'monthly'],
        ];

        foreach ($staticPages as $url => $settings) {
            $this->addUrl($url, $settings['priority'], $settings['changefreq']);
        }

        return $this;
    }

    public function addDynamicPages()
    {
        // Add news articles
        if (class_exists('\App\Models\News')) {
            $news = \App\Models\News::all();
            foreach ($news as $article) {
                // Generate slug from title if slug doesn't exist
                $slug = $article->slug ?? \Illuminate\Support\Str::slug($article->title);
                $this->addUrl(
                    "/news/{$slug}",
                    0.7,
                    'monthly',
                    $article->updated_at->toISOString()
                );
            }
        }

        // Add brand pages
        if (class_exists('\App\Models\Brand')) {
            $brands = \App\Models\Brand::all();
            foreach ($brands as $brand) {
                $slug = $brand->slug ?? \Illuminate\Support\Str::slug($brand->name);
                $this->addUrl(
                    "/brands/{$slug}",
                    0.8,
                    'monthly',
                    $brand->updated_at->toISOString()
                );
            }
        }

        return $this;
    }

    public function generate()
    {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($this->urls as $urlData) {
            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . $this->baseUrl . $urlData['url'] . '</loc>' . "\n";
            $xml .= '    <lastmod>' . $urlData['lastmod'] . '</lastmod>' . "\n";
            $xml .= '    <changefreq>' . $urlData['changefreq'] . '</changefreq>' . "\n";
            $xml .= '    <priority>' . $urlData['priority'] . '</priority>' . "\n";
            $xml .= '  </url>' . "\n";
        }

        $xml .= '</urlset>';

        return $xml;
    }

    public function save($path = null)
    {
        $path = $path ?: public_path('sitemap.xml');
        $xml = $this->generate();
        
        File::put($path, $xml);
        
        return $path;
    }

    public function generateRobotsTxt()
    {
        $robots = "User-agent: *\n";
        $robots .= "Allow: /\n\n";
        $robots .= "Sitemap: " . $this->baseUrl . "/sitemap.xml\n";
        
        return $robots;
    }

    public function saveRobotsTxt($path = null)
    {
        $path = $path ?: public_path('robots.txt');
        $robots = $this->generateRobotsTxt();
        
        File::put($path, $robots);
        
        return $path;
    }
} 