<?php

namespace App\Services;

use App\Models\News;
use Illuminate\Pagination\LengthAwarePaginator;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;

class NewsService
{
    public function getNews()
    {
        return Cache::remember('news.all', 60, function () {
            return News::latest()->take(7)->get();
        });
    }

    public function getNewsById($id)
    {
        return Cache::remember("news.{$id}", 60, function () use ($id) {
            return News::findOrFail($id);
        });
    }

    public function filterNews($searchQuery, $categoryQuery, $page)
    {
        $query = News::query();

        // Apply search filter
        if (!empty($searchQuery)) {
            $query->where(function($q) use ($searchQuery) {
                $q->where('title', 'like', "%{$searchQuery}%")
                  ->orWhere('body', 'like', "%{$searchQuery}%")
                  ->orWhere('tags', 'like', "%{$searchQuery}%");
            });
        }

        // Apply category filter
        if (!empty($categoryQuery) && $categoryQuery != "all") {
            $query->where('category', $categoryQuery);
        }

        // Get total count before pagination
        $totalNews = $query->count();

        // Apply pagination with proper date ordering
        $perPage = 9;
        $news = $query->orderBy('created_at', 'DESC')
                     ->skip(($page - 1) * $perPage)
                     ->take($perPage)
                     ->get();

        return [
            "news" => $news,
            "page" => $page,
            "totalNews" => $totalNews,
            "totalPages" => ceil($totalNews / $perPage),
            "searchQuery" => $searchQuery,
            "categoryQuery" => $categoryQuery,
        ];
    }

    public function searchNews($query)
    {
        return News::where(function($q) use ($query) {
            $q->where('title', 'like', "%{$query}%")
              ->orWhere('body', 'like', "%{$query}%")
              ->orWhere('category', 'like', "%{$query}%");
        })
        ->orderBy('created_at', 'desc')
        ->get();
    }
}
