<?php

namespace App\Http\Controllers;

use App\Models\SpamAttempt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class SpamMonitoringController extends Controller
{
    /**
     * Show spam monitoring dashboard
     */
    public function index(Request $request)
    {
        $days = $request->get('days', 7);
        $stats = SpamAttempt::getSpamStats($days);
        
        $recentAttempts = SpamAttempt::with([])
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get();
            
        $topSpamIps = SpamAttempt::selectRaw('ip_address, COUNT(*) as attempt_count')
            ->where('created_at', '>=', now()->subDays($days))
            ->groupBy('ip_address')
            ->orderBy('attempt_count', 'desc')
            ->limit(10)
            ->get();

        return view('admin.spam-monitoring', compact('stats', 'recentAttempts', 'topSpamIps', 'days'));
    }

    /**
     * Get spam statistics as JSON for AJAX requests
     */
    public function getStats(Request $request)
    {
        $days = $request->get('days', 7);
        $stats = SpamAttempt::getSpamStats($days);
        
        return response()->json($stats);
    }

    /**
     * Block an IP address
     */
    public function blockIp(Request $request)
    {
        $ip = $request->input('ip');
        
        if (!$ip) {
            return response()->json(['error' => 'IP address is required'], 400);
        }

        // Flag IP as spam
        $spamDetection = new \App\Services\SpamDetectionService();
        $spamDetection->flagSpamIp($ip);

        // Update database records
        SpamAttempt::where('ip_address', $ip)
            ->update(['blocked' => true, 'blocked_at' => now()]);

        return response()->json(['success' => true, 'message' => "IP {$ip} has been blocked"]);
    }

    /**
     * Unblock an IP address
     */
    public function unblockIp(Request $request)
    {
        $ip = $request->input('ip');
        
        if (!$ip) {
            return response()->json(['error' => 'IP address is required'], 400);
        }

        // Remove from cache
        $cacheKey = 'spam_ip_' . md5($ip);
        Cache::forget($cacheKey);

        // Update database records
        SpamAttempt::where('ip_address', $ip)
            ->update(['blocked' => false, 'blocked_at' => null]);

        return response()->json(['success' => true, 'message' => "IP {$ip} has been unblocked"]);
    }

    /**
     * Clear old spam attempts
     */
    public function clearOldAttempts(Request $request)
    {
        $days = $request->input('days', 30);
        
        $deleted = SpamAttempt::where('created_at', '<', now()->subDays($days))->delete();
        
        return response()->json([
            'success' => true, 
            'message' => "Deleted {$deleted} spam attempts older than {$days} days"
        ]);
    }
}
